<?php
// api/endpoints/orders.php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST, GET"); // Raporlar için GET de eklendi
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

include_once '../config/database.php';
include_once '../models/Order.php';
include_once '../models/Product.php';
include_once '../models/User.php'; // Kullanıcı rolünü kontrol etmek için

$database = new Database();
$db = $database->getConnection();

$order = new Order($db);
$product = new Product($db);
$user_model = new User($db);

// Yetkilendirme kontrolü için yardımcı fonksiyon
function authorize_manager($db_conn, $username) {
    $user_model_check = new User($db_conn);
    $user_model_check->username = $username;
    if ($user_model_check->getRoleByUsername() && $user_model_check->role === 'manager') {
        return true;
    }
    return false;
}

switch ($request_method) {
    case 'POST':
        // Yeni sipariş oluştur (Çalışanlar da yapabilir)
        $data = json_decode(file_get_contents("php://input"));

        if (
            !empty($data->total_amount) &&
            !empty($data->items)
        ) {
            $order->total_amount = $data->total_amount;
            $order->status = 'completed'; // Varsayılan olarak tamamlandı

            // Sipariş oluşturma işlemi bir transaction içinde yapılmalı
            $db->beginTransaction();

            try {
                if ($order->create()) {
                    $order_id = $order->id;

                    // Sipariş kalemlerini ekle
                    if ($order->createOrderItems($order_id, $data->items)) {
                        // Ürün stoklarını güncelle
                        foreach ($data->items as $item) {
                            // Stoktan düşme (negatif değer göndermeliyiz)
                            if (!$product->updateStock($item->product_id, -$item->quantity)) {
                                throw new Exception("Ürün stoğu güncellenemedi.");
                            }
                        }
                        $db->commit();
                        http_response_code(201);
                        echo json_encode(array("message" => "Sipariş başarıyla oluşturuldu."));
                    } else {
                        throw new Exception("Sipariş kalemleri oluşturulamadı.");
                    }
                } else {
                    throw new Exception("Sipariş oluşturulamadı.");
                }
            } catch (Exception $e) {
                $db->rollBack();
                http_response_code(503);
                echo json_encode(array("message" => "Sipariş işlemi başarısız oldu: " . $e->getMessage()));
            }

        } else {
            http_response_code(400);
            echo json_encode(array("message" => "Sipariş oluşturulamadı. Veri eksik."));
        }
        break;

    case 'GET':
        // Raporlara erişim sadece yöneticiler için
        $authorized_username = isset($_GET['username']) ? $_GET['username'] : ''; // URL'den username al

        if (!authorize_manager($db, $authorized_username)) {
            http_response_code(403); // Yasak
            echo json_encode(array("message" => "Raporlara erişim için yönetici yetkisi gereklidir."));
            exit();
        }

        // Yetkilendirme başarılı, şimdi raporları oluştur
        $period = isset($_GET['period']) ? $_GET['period'] : 'daily';
        $report_type = isset($_GET['report_type']) ? $_GET['report_type'] : 'sales';

        if ($report_type == 'sales') {
            $stmt = $order->getReports($period);
            $num = $stmt->rowCount();

            if ($num > 0) {
                $reports_arr = array();
                $reports_arr["records"] = array();

                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    array_push($reports_arr["records"], $row);
                }
                http_response_code(200);
                echo json_encode($reports_arr);
            } else {
                http_response_code(404);
                echo json_encode(array("message" => "Rapor bulunamadı."));
            }
        } elseif ($report_type == 'top_selling') {
            $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 5;
            $stmt = $order->getTopSellingProducts($limit);
            $num = $stmt->rowCount();

            if ($num > 0) {
                $top_selling_arr = array();
                $top_selling_arr["records"] = array();

                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    array_push($top_selling_arr["records"], $row);
                }
                http_response_code(200);
                echo json_encode($top_selling_arr);
            } else {
                http_response_code(404);
                echo json_encode(array("message" => "En çok satan ürün bulunamadı."));
            }
        } else {
            http_response_code(400);
            echo json_encode(array("message" => "Geçersiz rapor türü."));
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(array("message" => "Geçersiz istek metodu."));
        break;
}
?>
