<?php
// api/models/Order.php

class Order {
    private $conn;
    private $table_name = "orders";
    private $order_items_table = "order_items";

    public $id;
    public $total_amount;
    public $status;
    public $order_date;
    public $order_items; // Sipariş kalemlerini tutacak

    public function __construct($db) {
        $this->conn = $db;
    }

    // Yeni sipariş oluştur
    function create() {
        $query = "INSERT INTO " . $this->table_name . " SET total_amount=:total_amount, status=:status";
        $stmt = $this->conn->prepare($query);

        $this->total_amount = htmlspecialchars(strip_tags($this->total_amount));
        $this->status = htmlspecialchars(strip_tags($this->status));

        $stmt->bindParam(":total_amount", $this->total_amount);
        $stmt->bindParam(":status", $this->status);

        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId(); // Yeni oluşturulan siparişin ID'sini al
            return true;
        }
        return false;
    }

    // Sipariş kalemlerini (order_items) ekle
    function createOrderItems($order_id, $items) {
        foreach ($items as $item) {
            $query = "INSERT INTO " . $this->order_items_table . " SET order_id=:order_id, product_id=:product_id, quantity=:quantity, price=:price";
            $stmt = $this->conn->prepare($query);

            $product_id = htmlspecialchars(strip_tags($item->product_id));
            $quantity = htmlspecialchars(strip_tags($item->quantity));
            $price = htmlspecialchars(strip_tags($item->price)); // Ürün o anki fiyatı

            $stmt->bindParam(":order_id", $order_id);
            $stmt->bindParam(":product_id", $product_id);
            $stmt->bindParam(":quantity", $quantity);
            $stmt->bindParam(":price", $price);

            if (!$stmt->execute()) {
                return false;
            }
        }
        return true;
    }

    // Günlük veya aylık raporları al
    function getReports($period = 'daily') {
        $query = "";
        if ($period == 'daily') {
            $query = "SELECT DATE(order_date) as date, SUM(total_amount) as total_sales, COUNT(id) as total_orders FROM " . $this->table_name . " WHERE DATE(order_date) = CURDATE() AND status = 'completed' GROUP BY DATE(order_date)";
        } elseif ($period == 'monthly') {
            $query = "SELECT DATE_FORMAT(order_date, '%Y-%m') as month, SUM(total_amount) as total_sales, COUNT(id) as total_orders FROM " . $this->table_name . " WHERE status = 'completed' GROUP BY DATE_FORMAT(order_date, '%Y-%m') ORDER BY month DESC";
        } elseif ($period == 'all') { // Geriye dönük tüm raporlar
             $query = "SELECT DATE_FORMAT(order_date, '%Y-%m-%d') as date, SUM(total_amount) as total_sales, COUNT(id) as total_orders FROM " . $this->table_name . " WHERE status = 'completed' GROUP BY DATE(order_date) ORDER BY date DESC";
        }

        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }

    // En çok satan ürünleri al
    function getTopSellingProducts($limit = 5) {
        $query = "SELECT p.name, SUM(oi.quantity) as total_quantity_sold
                  FROM " . $this->order_items_table . " oi
                  LEFT JOIN products p ON oi.product_id = p.id
                  LEFT JOIN orders o ON oi.order_id = o.id
                  WHERE o.status = 'completed'
                  GROUP BY p.name
                  ORDER BY total_quantity_sold DESC
                  LIMIT :limit";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt;
    }
}
?>
